<?php
// src/TelegramClient.php - minimal Telegram API wrapper using cURL with proxy support
class TelegramClient {
    private $token;
    private $api_base;
    private $proxy_enabled;
    private $proxy_url;
    public function __construct($token, $api_base='https://api.telegram.org', $proxy_enabled=false, $proxy_url=null){
        $this->token = $token;
        $this->api_base = rtrim($api_base, '/');
        $this->proxy_enabled = filter_var($proxy_enabled, FILTER_VALIDATE_BOOLEAN);
        $this->proxy_url = $proxy_url;
    }
    private function request($method, $params = [], $isMultipart=false){
        $url = $this->api_base.'/bot'.$this->token.'/'.$method;
        $ch = curl_init();
        if($isMultipart){
            curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: multipart/form-data']);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $params);
        } else {
            $query = http_build_query($params);
            $url .= '?'.$query;
        }
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        // proxy support
        if($this->proxy_enabled && !empty($this->proxy_url)){
            curl_setopt($ch, CURLOPT_PROXY, $this->proxy_url);
            // optionally set proxy type: HTTP by default, one can change to SOCKS5 if needed
        }
        $resp = curl_exec($ch);
        if($resp === false){
            $err = curl_error($ch);
            curl_close($ch);
            throw new Exception('Telegram request failed: '.$err);
        }
        curl_close($ch);
        return json_decode($resp, true);
    }
    public function sendMessage($chat_id, $text, $reply_markup=null){
        $params = ['chat_id'=>$chat_id, 'text'=>$text, 'parse_mode'=>'HTML'];
        if($reply_markup) $params['reply_markup'] = json_encode($reply_markup);
        return $this->request('sendMessage', $params);
    }
    public function sendChatAction($chat_id, $action='typing'){
        return $this->request('sendChatAction', ['chat_id'=>$chat_id,'action'=>$action]);
    }
    public function sendFileByPath($chat_id, $type, $file_path, $caption=null){
        // type: 'photo','audio','video','document','voice'
        $paramName = ($type === 'photo') ? 'photo' : $type;
        if(!file_exists($file_path)) throw new Exception('File not found: '.$file_path);
        $params = ['chat_id'=>$chat_id, $paramName=>new CURLFile($file_path)];
        if($caption) $params['caption'] = $caption;
        return $this->request('send'.$this->ucfirstType($type), $params, true);
    }
    private function ucfirstType($t){
        // map to Telegram method suffix
        $map = ['photo'=>'Photo','audio'=>'Audio','video'=>'Video','document'=>'Document','voice'=>'Voice'];
        return $map[$t] ?? ucfirst($t);
    }
}
